/********************************************************
 *   ____                           _    ____            *
 *  / ___|___  _ __   ___ ___ _ __ | |_ / ___|__ _ _ __  *
 * | |   / _ \| '_ \ / __/ _ \ '_ \| __| |   / _` | '__| *
 * | |__| (_) | | | | (_|  __/ |_) | |_| |__| (_| | |    *
 *  \____\___/|_| |_|\___\___| .__/ \__|\____\__,_|_|    *
 *   Copyright 2005-2009     |_|  Fraunhofer IESE KL     *
 *                                                       *
 *                                                       *
 * @version 0.4	                                         *
 * @date 2009-08                                         *
 *********************************************************/

/**
 * This is a CAN driver library for the AT91SAM7A2.
 * It is to be run directly on the bare hardware, and manages
 * all initialization and device handling.
 * The device is initialized to run at 1MBit/s CAN highspeed
 * with protocol 2.0A.
 * From the ARM7, only the first CAN module (module 0) is
 * used with all its 16 channels.
 * Messages other than 4 byte length are ignored.
 */
#include "can.h"
#include "canids.h"

#include "at91sam7a2_interrupts.h"
#include "at91sam7a2_timers.h"
#include <string.h>
#include "util.h"
#include "global.h"
#include <inttypes.h>


/**
 * The number of channels reserved for receiving.
 * Must be >0 and <13. The remaining of the 3 channels are reserved
 * for sending.
 */
#define CAN_RECEIVE_CHANNELS 12
#define CAN_BUFFER_SIZE 16
/** The internal receive buffer */
CAN_BUFFER canBuffer[CAN_BUFFER_SIZE];
/** A static link to the control registers of module 0 */
AT91SAM7A2_CAN_MODULE* can;

/**
 * The Interrupt service routine for handling the CAN0-Interrupt.
 *
 * Reads messages to the receive buffer.
 */
__attribute__((interrupt("IRQ"))) void _canISR() {
	if (can->CAN_SR & (1 << CAN_ISS)) {
		//a channel has generated an interrupt
		// find responsible channel
		int channelNr;
		int length = 0;
		for (channelNr = 0; channelNr < CAN_RECEIVE_CHANNELS; channelNr++) //CAN0_MAX_CHANNELS
		if (can->CAN_ISSR & (1 << channelNr))
		break; // channel caused the interrupt
		AT91SAM7A2_CAN_CHANNEL* ch = &(can->ch[channelNr]);

		// is new data available?
		if (ch->CAN_SR & (1 << CAN_RXOK)) {
			// does it have an allowed size?
			canBuffer[channelNr].bufferHigh = ch->CAN_DRB;
			canBuffer[channelNr].bufferLow = ch->CAN_DRA;
			canBuffer[channelNr].length = length;
			canBuffer[channelNr].ready = 1;
			canBuffer[channelNr].ts = ch->CAN_STP;
			}


		// clear channels status register
		ch->CAN_CSR = (1 << CAN_ACK) | (1 << CAN_FRAME) | (1 << CAN_CRC)
					| (1 << CAN_STUFF) | (1 << CAN_BUS) | (1 << CAN_RXOK) | (1
					<< CAN_TXOK) | (1 << CAN_RFRAME) | (1 << CAN_DLCW) | (1
					<< CAN_FILLED) | (1 << CAN_OVRUN); // clear all possible interrupts
		//finally, clear channel as interrupt source
		can->CAN_CISR = (1 << channelNr);
		at91sam7a2_end_of_interrupt();

	}
	if (can->CAN_SR & (1 << CAN_ERPAS)) {
		// changed to error passive
		// TODO
	}
	if (can->CAN_SR & (1 << CAN_BUSOFF)) {
		// changed to error passive
		// TODO
	}
}


/**
 * Initialization routine for the CAN module 0.
 * This function must be called before the CAN driver can be
 * used.
 */
void can_init(void) {
	// clear can receive buffer
	memset(canBuffer, 0, sizeof(canBuffer));

	can = (AT91SAM7A2_CAN_MODULE*) CAN0_BASE_ADDRESS;
	// enable CAN clock
	// (CORECLK=30MHz, CANCLK=3MHz)
	can->CAN_ECR = 1 << CAN_ENABLE;

	// CAN software reset
	can->CAN_CR = 1 << CAN_SWRST;

	// wait for register initialization
	while (!(can->CAN_SR & (1 << CAN_ENDINIT)))
		;

	// configure CAN timings
	can->CAN_MR = (PHSEG2 << CAN_PHSEG2) | (PHSEG1 << CAN_PHSEG1) | (1
			<< CAN_SMP) | (SJW << CAN_SJW) | (PROP << CAN_PROP) | (BD);

	// enable CAN bus
	can->CAN_CR = 1 << CAN_CANEN;

	// wait for can being actually enabled
	while (!(can->CAN_SR & (1 << CAN_CANENA)))
		;

}

//This routine must be used before calling the receive routine, setting up the channel beforehand
void configure_can_channel(unsigned int channel_nr, unsigned int id, unsigned int mask, short interrupt_mode)
{
		AT91SAM7A2_CAN_CHANNEL* ch = &(can->ch[channel_nr]);

		//enable can interrupt for receiving messages and moving to buffer
		ch->CAN_IR = id;
		ch->CAN_MSK = mask; // receive message only with this id
		ch->CAN_CR = 0x00000004ul;
		ch->CAN_CR = ((1 << CAN_CHANEN) | (1 <<CAN_OVERWRITE)); //  enable as receiver 2.0A

		if(interrupt_mode)
		{
			ch->CAN_IER = (1 << CAN_RXOK); // enable receive interrupt	//my test
			can->CAN_SIER = (1 << channel_nr); // enable channel interrupt

			//enable can interrupt for receiving messages and moving to buffer
			at91sam7a2_interrupts_configure_source(_canISR, IRQ_CAN0);	//my test
			at91sam7a2_interrupts_enable(IRQ_CAN0);						//my test

		}
}

int can_receive(unsigned int channel_nr, unsigned int* buf, short interrupt_mode) {
	AT91SAM7A2_CAN_CHANNEL* ch = &(can->ch[channel_nr]);

				// is new data available?

				if(interrupt_mode)
				{

					if((canBuffer[channel_nr].ready == 1))
					{
						*buf = canBuffer[channel_nr].bufferLow;
						canBuffer[channel_nr].ready = 0;
						return(1);

					}

					else
						return(0);

				}

				else {
					while(!(ch->CAN_SR & (1 << CAN_RXOK)));
					*buf = ch->CAN_DRA;
					ch->CAN_CSR = 1 << CAN_RXOK;
				}

	return 1;
}



/**
 * Send a message with the given ID.
 * Searches for some time for a free channel for sending, then aborts.
 * @param id The can ID to send at
 * @param data The data to send
 * @return Did writing the data to the transmit buffer of a free channel succeed?
 */
int can_send(unsigned int id, unsigned int data) {
	// do some iterations before giving up...
		for (int i = 0; i < 10000; i++) {
			// select a free channel for sending
			for (int i = CAN_RECEIVE_CHANNELS; i < CAN0_MAX_CHANNELS; i++) {
				AT91SAM7A2_CAN_CHANNEL* ch = &(can->ch[i]);
				if (ch->CAN_CR & (1 << CAN_CHANEN))
					continue; // channel is busy


			// found a free channel
			ch->CAN_DRA = data;
			ch->CAN_DRB = 0;
			ch->CAN_IR = 0x7ff & id;
			ch->CAN_CR = (1 << CAN_CHANEN) | (1 << CAN_PCB) | 4; // send 4 byte
			return 1; // all done
			/* optional: active wait for transmit finish
			 while (! (ch->CAN_SR & (1 << CAN_TXOK)));
			 ch->CAN_CSR = (1 << CAN_TXOK);
			 */

			}
		}
	// failed to send: all channels busy
	return 0;
}

void Configure_CAN_Channels()
{

	configure_can_channel(CHANNEL0, CANID_WHEELSPEED_FR   , MASK, INTERRUPT_MODE);
	configure_can_channel(CHANNEL1, CANID_WHEELSPEED_FL   , MASK, INTERRUPT_MODE);
	configure_can_channel(CHANNEL2, CANID_WHEELSPEED_RR   , MASK, INTERRUPT_MODE);
	configure_can_channel(CHANNEL3, CANID_WHEELSPEED_FL   , MASK, INTERRUPT_MODE);
	configure_can_channel(CHANNEL4, CANID_THROTTLE  , MASK, INTERRUPT_MODE);
	configure_can_channel(CHANNEL5, CANID_STEERING  , MASK, INTERRUPT_MODE);
	configure_can_channel(CHANNEL6, CANID_INERTIAL_AX  , MASK, INTERRUPT_MODE);
	configure_can_channel(CHANNEL7, CANID_INERTIAL_AY  , MASK, INTERRUPT_MODE);
	configure_can_channel(CHANNEL8, CANID_QUARTZ_INPUT1  , MASK, INTERRUPT_MODE);
	configure_can_channel(CHANNEL9, CANID_QUARTZ_INPUT2 , MASK, INTERRUPT_MODE);
	configure_can_channel(CHANNEL10, CANID_QUARTZ_INPUT3 , MASK, INTERRUPT_MODE);
}

/*Functions for SpeedMeter Application*/
float calculate_time_duration(unsigned int _wheel_speed_data)
{
	float _time = 0.0f;
	uint8_t low_byte = *(((uint8_t*) &_wheel_speed_data) + 0);
	uint8_t hi_byte = *(((uint8_t*) &_wheel_speed_data) + 1);
	uint8_t overflow = *(((uint8_t*) &_wheel_speed_data) + 2);
	_time = ((overflow*65536) + (hi_byte*256) + low_byte);

	return _time;
}

uint32_t encode_speed_data(float speed_ms)
{
	uint32_t speed_ms_encoded = 0;
	memcpy(&speed_ms_encoded, &speed_ms, sizeof(speed_ms_encoded));

	return speed_ms_encoded;
}

