/********************************************************
 *   ____                           _    ____            *
 *  / ___|___  _ __   ___ ___ _ __ | |_ / ___|__ _ _ __  *
 * | |   / _ \| '_ \ / __/ _ \ '_ \| __| |   / _` | '__| *
 * | |__| (_) | | | | (_|  __/ |_) | |_| |__| (_| | |    *
 *  \____\___/|_| |_|\___\___| .__/ \__|\____\__,_|_|    *
 *   Copyright 2005-2009     |_|  Fraunhofer IESE KL     *
 *                                                       *
 *                                                       *
 * @version 0.4	                                         *
 * @date 2009-08                                         *
 *********************************************************/

/**
 * This is a CAN driver library for the AT91SAM7A2.
 * It is to be run directly on the bare hardware, and manages
 * all initialization and device handling.
 * The device is initialized to run at 1MBit/s CAN highspeed
 * with protocol 2.0A.
 * From the ARM7, only the first CAN module (module 0) is
 * used with all its 16 channels.
 * Messages other than 4 byte length are ignored.
 */
#include "can.h"

#include "at91sam7a2_interrupts.h"
#include <string.h>

/**
 * The size of the receive buffer, one buffer line for each distinct
 * can ID to receive is required.
 */
#define CAN_BUFFER_SIZE	128
/**
 * The number of channels reserved for receiving.
 * Must be >0 and <16. The remaining of the 16 channels are reserved
 * for sending.
 */
#define CAN_RECEIVE_CHANNELS 9
/** The internal receive buffer */
CAN_BUFFER canBuffer[CAN_BUFFER_SIZE];
/** A static link to the control registers of module 0 */
AT91SAM7A2_CAN_MODULE* can;

/**
 * The Interrupt service routine for handling the CAN0-Interrupt.
 *
 * Reads messages to the receive buffer.
 */
void _canISR() {
	if (can->CAN_SR & (1 << CAN_ISS)) {
		//a channel has generated an interrupt
		while (can->CAN_ISSR) {
			// work on channel interrupts until they are all cleared

			// find responsible channel
			int channelNr;
			for (channelNr = 1; channelNr < CAN_RECEIVE_CHANNELS; channelNr++) //CAN0_MAX_CHANNELS
				if (can->CAN_ISSR & (1 << channelNr))
					break; // channel caused the interrupt
			AT91SAM7A2_CAN_CHANNEL* ch = &(can->ch[channelNr]);

			// is new data available?
			if (ch->CAN_SR & (1 << CAN_RXOK)) {
				// does it have an allowed size?
				unsigned int length = ch->CAN_CR & 0xf;
				if (length == 4) {
					// does it match a registered can id?
					unsigned int id = ch->CAN_IR;
					for (int i = 0; i < CAN_BUFFER_SIZE; i++) {
						if (canBuffer[i].id == id) {
							// message is to be received
							canBuffer[i].bufferHigh = ch->CAN_DRB;
							canBuffer[i].bufferLow = ch->CAN_DRA;
							canBuffer[i].length = length;
							canBuffer[i].ready = 1;
							canBuffer[i].ts = ch->CAN_STP;

							break; // receive done
						}
						if (canBuffer[i].id == 0)
							break; // end of buffer, not found
					}
				}
				// re-enable for next receive
				ch->CAN_CR = (1 << CAN_CHANEN);
			}

			// clear channels status register
			ch->CAN_CSR = (1 << CAN_ACK) | (1 << CAN_FRAME) | (1 << CAN_CRC)
					| (1 << CAN_STUFF) | (1 << CAN_BUS) | (1 << CAN_RXOK) | (1
					<< CAN_TXOK) | (1 << CAN_RFRAME) | (1 << CAN_DLCW) | (1
					<< CAN_FILLED) | (1 << CAN_OVRUN); // clear all possible interrupts
			// finally, clear channel as interrupt source
			can->CAN_CISR = (1 << channelNr);
		}
	}
	if (can->CAN_SR & (1 << CAN_ERPAS)) {
		// changed to error passive
		// TODO
	}
	if (can->CAN_SR & (1 << CAN_BUSOFF)) {
		// changed to error passive
		// TODO
	}
}

/**
 * Initialization routine for the CAN module 0.
 * This function must be called before the CAN driver can be
 * used.
 */
void can_init(void) {
	// clear can receive buffer
	memset(canBuffer, 0, sizeof(canBuffer));

	can = (AT91SAM7A2_CAN_MODULE*) CAN0_BASE_ADDRESS;
	// enable CAN clock
	// (CORECLK=30MHz, CANCLK=3MHz)
	can->CAN_ECR = 1 << CAN_ENABLE;

	// CAN software reset
	can->CAN_CR = 1 << CAN_SWRST;

	// wait for register initialization
	while (!(can->CAN_SR & (1 << CAN_ENDINIT)))
		;

	// configure CAN timings
	can->CAN_MR = (PHSEG2 << CAN_PHSEG2) | (PHSEG1 << CAN_PHSEG1) | (1
			<< CAN_SMP) | (SJW << CAN_SJW) | (PROP << CAN_PROP) | (BD);

	// enable CAN bus
	can->CAN_CR = 1 << CAN_CANEN;

	// wait for can being actually enabled
	while (!(can->CAN_SR & (1 << CAN_ENDINIT)))
		;

	// enable and configure all receiving channels
	can->CAN_SIDR = 0xfffffffful;
	for (int i = 1; i < CAN_RECEIVE_CHANNELS; i++) {
		AT91SAM7A2_CAN_CHANNEL* ch = &(can->ch[i]);
		ch->CAN_CR = 0x00000004ul;
		ch->CAN_MSK = 0x000003fful; // receive all incoming messages
		ch->CAN_IR = 0x00000000ul;
		ch->CAN_STP = 0x00000000ul;
		ch->CAN_CSR = (1 << CAN_ACK) | (1 << CAN_FRAME) | (1 << CAN_CRC) | (1
				<< CAN_STUFF) | (1 << CAN_BUS) | (1 << CAN_RXOK) | (1
				<< CAN_TXOK) | (1 << CAN_RFRAME) | (1 << CAN_DLCW) | (1
				<< CAN_FILLED) | (1 << CAN_OVRUN); // clear all possible interrupts
		ch->CAN_CR = (1 << CAN_CHANEN); // enable as receiver 2.0A
		ch->CAN_IER = (1 << CAN_RXOK); // enable receive interrupt	//my test

		can->CAN_SIER = (1 << i); // enable channel interrupt
	}
	// initialize all channels for sending
	for (int i = CAN_RECEIVE_CHANNELS; i < CAN0_MAX_CHANNELS; i++) {
		AT91SAM7A2_CAN_CHANNEL* ch = &(can->ch[i]);
		ch->CAN_CR = 0x00000000ul;
	}

	//enable can interrupt for receiving messages and moving to buffer
	at91sam7a2_interrupts_configure_source(_canISR, IRQ_CAN0);	//my test
	at91sam7a2_interrupts_enable(IRQ_CAN0);						//my test

	// enable bus off and error passive interrupts
	can->CAN_IER = (1 << CAN_ERPAS) | (1 << CAN_BUSOFF);			//my test
}

/**
 * Register a can ID for receiving.
 * No message can be received before its ID is registered.
 * @param The can ID that should be received from now on.
 */
void can_subscribeID(unsigned int id) {
	for (int i = 0; i < CAN_BUFFER_SIZE; i++) {
		if (canBuffer[i].id == id)
			return;
		if (canBuffer[i].id == 0) {
			canBuffer[i].id = id;
			return;
		}
	}
}

/**
 * Check if the can receive buffer contains a message and fetch it.
 * Before this works, the message's can ID must be registered.
 * @param id The can ID to check
 * @param buf A buffer the message's content is copied to
 * @return Did the buffer hold a message for this ID?
 */
int can_receive(unsigned int id, unsigned int* buf) {
	for (int i = 0; i < CAN_BUFFER_SIZE; i++) {
		if (canBuffer[i].id == 0)
			return 0;
		if ((canBuffer[i].id == id) &&
				(canBuffer[i].ready)) {
			*buf = canBuffer[i].bufferLow;
			//*buf = ch->CAN_DRA;  //my test
			canBuffer[i].ready = 0;
			return 1;
		}
	}
	return 0;
}

/**
 * Retrieve the latest timestamp for messages with the given ID.
 * @param id The can ID to check for
 * @return The timestamp of the last message with this id, 0 if none found
 */
unsigned long can_receiveTimestamp(unsigned int id) {
	for (int i = 0; i < CAN_BUFFER_SIZE; i++) {
		if (canBuffer[i].id == 0)
			return 0;
		if (canBuffer[i].id == id) {
			return canBuffer[i].ts;
		}
	}
	return 0;
}

/**
 * Send a message with the given ID.
 * Searches for some time for a free channel for sending, then aborts.
 * @param id The can ID to send at
 * @param data The data to send
 * @return Did writing the data to the transmit buffer of a free channel succeed?
 */
int can_send(unsigned int id, unsigned int data) {
	// do some iterations before giving up...
	for (int i = 0; i < 10000; i++) {
		// select a free channel for sending
		for (int i = CAN_RECEIVE_CHANNELS; i < CAN0_MAX_CHANNELS; i++) {
			AT91SAM7A2_CAN_CHANNEL* ch = &(can->ch[i]);
			if (ch->CAN_CR & (1 << CAN_CHANEN))
				continue; // channel is busy

			// found a free channel
			ch->CAN_DRA = data;
			ch->CAN_DRB = 0;
			ch->CAN_IR = 0x7ff & id;
			ch->CAN_CR = (1 << CAN_CHANEN) | (1 << CAN_PCB) | 4; // send 4 byte
			return 1; // all done
			/* optional: active wait for transmit finish
			 while (! (ch->CAN_SR & (1 << CAN_TXOK)));
			 ch->CAN_CSR = (1 << CAN_TXOK);
			 */
		}
	}
	// failed to send: all channels busy
	return 0;
}

